/**
 * Created by Emil Riedeman on 2024-12-27
 */

/*******************************************************************************
 * ZeroPointOne is the object which manages the game visualisation.
 * This part makes sure that the board transitions can be made.
 * When the board is changed, it sends the board to all its controllers.
 ******************************************************************************/

function createPiece(row, col, color, type) {
	const div = document.createElement('div');
	div.classList.add('piece');
	div.classList.add('on-board');
	div.classList.add(color);
	div.classList.add(`piece-${type.toLowerCase()}`);
	div.style.setProperty('--_row', row);
	div.style.setProperty('--_col', col);
	return div;
}

function ZeroPointOne(playerNames, moves) {
	this.playerNames = playerNames;
	this.isRunning = false;
	this.curMove = 0;
	this.H = 8;
	this.W = 8;
	this.colors = [ "red", "blue" ];
	this.capturedPieces = [ [], [] ];

	this.moves = moves;

	this.registerEvents();
	this.buildMoveTable();

	this.initBoardTable();
	this.clearPieces();

	this.updateSelectedRow();

	this.currentMoveTransition = null;
}

ZeroPointOne.prototype.initBoardTable = function() {
	const gameContainer = document.getElementById('game-container');
	const table = document.getElementById('game-grid');

	gameContainer.style.setProperty('--_board-height', this.H);
	gameContainer.style.setProperty('--_board-width', this.W);

	// head
	const thead = table.createTHead();
	const headerRow = thead.insertRow();

	const cornerCell = document.createElement('th');
	cornerCell.textContent = '';
	headerRow.appendChild(cornerCell);

	for (let col = 0; col < this.W; col++) {
		const th = document.createElement('th');
		th.scope = 'col';
		th.textContent = String.fromCharCode(49 + col);
		headerRow.appendChild(th);
	}

	// body
	const tbody = table.createTBody();

	for (let row = 0; row < this.H; row++) {
		const rowElement = tbody.insertRow();
		const rowHeader = document.createElement('th');
		rowHeader.scope = 'row';
		rowHeader.textContent = String.fromCharCode(97 + row);
		rowElement.appendChild(rowHeader);

		for (let col = 0; col < this.W; col++) {
			const cell = rowElement.insertCell();
			cell.textContent = '';
		}
	}
};

ZeroPointOne.prototype.clearPieces = function() {
	if (this.currentMoveTransition) this.currentMoveTransition();

	document.getElementById('pieces').replaceChildren();

	this.pieces = {};

	this.capturedPieces = [ [], [] ];
}

ZeroPointOne.prototype.playerInit = function(initString) {
	const side = (initString.toLowerCase() == initString) ? 1 : 0;
	const rowBegin = [0, this.H - 2][side];
	const color = this.colors[side];

	for (let i = 0; i < this.W * 2; i++) {
		const type = initString[i];

		const row = Math.floor(i / this.W) + rowBegin;
		const col = i % this.W;

		const piece = createPiece(row, col, color, type);
		this.pieces[String.fromCharCode(row + 97) + String.fromCharCode(col + 49)] = piece;
		document.getElementById('pieces').appendChild(piece);
	}
}

ZeroPointOne.prototype.movePieceToCaptured = function(capturedPiece, transition) {
	if (!capturedPiece) return;

	// todo: revive transition: flip and scale + translate
	const side = capturedPiece.classList.contains('red') ? 0 : 1;
	capturedPiece.classList.remove(this.colors[side], 'on-board');
	capturedPiece.classList.add(this.colors[1 - side], 'captured');
	capturedPiece.style.setProperty('--_captured-index', this.capturedPieces[1 - side].length);
	this.capturedPieces[1 - side].push(capturedPiece);
}

ZeroPointOne.prototype.movePiece = function(move, transition) {
	let piece = this.pieces[move.substr(0, 2)];
	delete this.pieces[move.substr(0, 2)];
	const row = move.charCodeAt(2) - 97;
	const col = move.charCodeAt(3) - 49;

	piece.style.setProperty('--_row', row);
	piece.style.setProperty('--_col', col);

	let capturedPiece = this.pieces[move.substr(2, 4)];
	this.pieces[move.substr(2, 4)] = piece;

	if (this.currentMoveTransition) this.currentMoveTransition();

	if (!transition) {
		this.movePieceToCaptured(capturedPiece, false);
	} else {
		piece.classList.add('move-transition');

		let self = this;
		function newMoveTransition (e) {
			self.currentMoveTransition = null;
			piece.classList.remove('move-transition');
			piece.removeEventListener("transitionend", newMoveTransition);
			self.movePieceToCaptured(capturedPiece, true);
		};

		piece.addEventListener("transitionend", newMoveTransition);
		this.currentMoveTransition = newMoveTransition;
	}
}

ZeroPointOne.prototype.revivePiece = function(type, location, transition) {
	const side = (type.toLowerCase() == type) ? 1 : 0;
	const row = location.charCodeAt(0) - 97;
	const col = location.charCodeAt(1) - 49;
	const array = this.capturedPieces[side];
	let revivedPiece = null;

	// search captured piece with type:
	for (let i = array.length - 1; i >= 0; i--) {
		if (array[i].classList.contains(`piece-${type.toLowerCase()}`)) {
			revivedPiece = array.splice(i, 1)[0];
			this.reIndexCapturedPieces(side);
			break;
		}
	};

	revivedPiece.classList.remove('captured');
	revivedPiece.classList.add('on-board');
	revivedPiece.style.setProperty('--_row', row);
	revivedPiece.style.setProperty('--_col', col);
	this.pieces[location] = revivedPiece;

	if (this.currentMoveTransition) this.currentMoveTransition();

	if (transition) {
		revivedPiece.classList.add('move-transition');

		let self = this;
		function newMoveTransition (e) {
			self.currentMoveTransition = null;
			revivedPiece.classList.remove('move-transition');
			revivedPiece.removeEventListener("transitionend", newMoveTransition);
		};

		revivedPiece.addEventListener("transitionend", newMoveTransition);
		this.currentMoveTransition = newMoveTransition;
	}
}

ZeroPointOne.prototype.reIndexCapturedPieces = function(side) {
	for (const [index, capturedPiece] in this.capturedPieces[side].entries()) {
		capturedPiece.style.setProperty('--_captured-index', index);
	}
}

ZeroPointOne.prototype.doSingleMove = function(transition) {
	if (this.curMove >= this.moves.length - 1) return false;
	const move = this.moves[this.curMove + 1];

	// warning: hardcoded 1-8
	if (/^[a-z][1-8][a-z][1-8]$/.test(move)) {
		this.movePiece(move, transition);
	}
	else if (/^[wWnNfFdDaA][a-z][1-8]$/.test(move)) {
		this.revivePiece(move.substr(0, 1), move.substr(1, 3), transition);
	}
	else if (move.length == this.W * 2) {
		this.playerInit(move);
	}
	else if (this.curMove >= 2) {
		return false;
	}

	this.curMove++;
	this.updateSelectedRow();

	return true;
}

// returns whether the move has been processed
ZeroPointOne.prototype.setMove = function(moveId) {
	if (moveId >= this.moves.length || moveId < 0) return false;

	if (moveId === this.curMove + 1) {
		this.doSingleMove(true);
		return true;
	}

	this.clearPieces();

	for (this.curMove = 0; this.curMove < moveId; this.doSingleMove(false));

	this.updateSelectedRow();
	
	return true;
};

// -----------------------------------------------------------------------------
// Event handlers:
// -----------------------------------------------------------------------------

ZeroPointOne.prototype.setRunning = function(running) {
	if (this.isRunning == running) return;

	this.isRunning = running;
	this.playPause.innerText = running ? 'Pause' : 'Play';

	let self = this, run = function() {
		if (!self.isRunning) return;

		if (self.doSingleMove(true)) {
			setTimeout(run, 500);
		} else {
			// last move has been reached.
			self.setRunning(false);
		}
	}
	if (this.isRunning) run();
}

ZeroPointOne.prototype.prevMove = function() { this.setRunning(false); this.setMove(this.curMove - 1); };
ZeroPointOne.prototype.nextMove = function() { this.setRunning(false); this.doSingleMove(true); };

ZeroPointOne.prototype.registerEvents = function() {
	this.playPause = document.getElementById('play-pause');

	let self = this;
	document.getElementById('first').onclick = function() { self.setRunning(false); self.setMove(0); };
	document.getElementById('last').onclick = function() { self.setRunning(false); self.setMove(self.moves.length - 1); };
	document.getElementById('prev').onclick = function() { self.prevMove(); };
	document.getElementById('next').onclick = function() { self.nextMove(); };
	this.playPause.onclick = function() { self.setRunning(!self.isRunning); };
};

ZeroPointOne.prototype.buildMoveTable = function() {
	let C = function(tag, kids, attrs) {
		let x = document.createElement(tag);
		for (kid in (kids||[])) x.appendChild(kids[kid]);
		if (attrs) {
			if (attrs['class']) x.className = attrs['class'];
			if (attrs['text']) x.innerText = attrs['text'];
			if (attrs['colspan']) x.setAttribute('colspan', attrs['colspan']);
		}
		return x;
	};

	let self = this, nplayers = 2;
	this.moveTable = document.getElementById('move-table').children[0];

	let head = [ C('th') ];
	for (let p = 0; p < nplayers; p++) {
		// cut-off on more than 20 characters in the name
		let playerName = "Player " + (p + 1).toString();
		if (this.playerNames) {
			let name = this.playerNames[p];
			if (name.length > 15) name = name.slice(0, 12) + "...";
			playerName = name;
		}
		head.push(C('th', [], {text: playerName}));
	}

	let firstrow = C('tr', [ C('th', [], {text: '0.'}), C('td', [], {text: "Start"}) ]);
	firstrow.onclick = function() { self.setMove(0); };
	let rows = [firstrow];

	for (let i = 1; i < self.moves.length; i++) {
		let row = [ C('th', [], {text: i + '.'}) ], curp = (i + 1) % 2;
		for (let p = 0; p < nplayers; p++) {
			if (p == curp) row.push(C('td', [], {text: self.moves[i]}));
			else row.push(C('td'));
		}
		let tr = C('tr', row, {class: i % 2 ? 'zebra' : ''});
		tr.onclick = (function(move) { return function() { self.setMove(move); } })(i);
		rows.push(tr);
	}

	this.moveTableBody = C('tbody', rows);
	this.moveTable.onkeydown = function(event) {
		if (event.keyCode == 38 || event.keyCode == 37) {
			self.prevMove();
		} else if (event.keyCode == 40 || event.keyCode == 39) {
			self.nextMove();
		} else {
			return;
		}
		event.preventDefault();
		event.stopImmediatePropagation();
	};

	this.moveTable.appendChild(C('table', [C('thead', [C('tr', head)]), this.moveTableBody]));
};

ZeroPointOne.prototype.updateSelectedRow = function() {
	let lastCurMove = document.getElementById('cur-move');
	let tbl = this.moveTable;

	// reset the id.
	if (lastCurMove) lastCurMove.id = '';

	if (this.curMove >= 0) {
		let curRow = this.moveTableBody.children[this.curMove];
		curRow.id = 'cur-move';

		let topCoord = curRow.offsetTop;
		let bottomCoord = curRow.offsetTop + curRow.offsetHeight - tbl.offsetHeight;
		if (tbl.scrollTop < bottomCoord) {
			tbl.scrollTop = bottomCoord;
		} else if (tbl.scrollTop > topCoord) {
			tbl.scrollTop = topCoord;
		}
	} else {
		tbl.scrollTop = 0;
	}
};
